<?php
// upload.php — simpan foto dari HP (input file atau WebRTC) + normalisasi orientasi EXIF
// Requirement: ekstensi PHP exif & gd (atau imagick jika mau).

// Konfigurasi dasar
$allowed = ['image/jpeg','image/png','image/webp'];
$max_size = 15 * 1024 * 1024; // 15 MB
$uploadDir = __DIR__ . '/uploads';
$publicPrefix = 'uploads'; // URL relatif

if (!is_dir($uploadDir)) { mkdir($uploadDir, 0755, true); }

header('Content-Type: application/json; charset=utf-8');

if (!isset($_FILES['photo'])) {
  http_response_code(400);
  echo json_encode(['ok'=>false, 'error'=>'No file']);
  exit;
}

$f = $_FILES['photo'];
if ($f['error'] !== UPLOAD_ERR_OK) {
  http_response_code(400);
  echo json_encode(['ok'=>false, 'error'=>'Upload error code '.$f['error']]);
  exit;
}

if ($f['size'] <= 0 || $f['size'] > $max_size) {
  http_response_code(400);
  echo json_encode(['ok'=>false, 'error'=>'Invalid size']);
  exit;
}

// Validasi MIME dari tmp_name (lebih aman)
$mime = mime_content_type($f['tmp_name']);
if (!in_array($mime, $allowed)) {
  http_response_code(400);
  echo json_encode(['ok'=>false, 'error'=>'Type not allowed: '.$mime]);
  exit;
}

$extMap = [
  'image/jpeg' => 'jpg',
  'image/png'  => 'png',
  'image/webp' => 'webp',
];
$ext = $extMap[$mime] ?? pathinfo($f['name'], PATHINFO_EXTENSION);
$fname = 'cam_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
$dest = $uploadDir . '/' . $fname;

if (!move_uploaded_file($f['tmp_name'], $dest)) {
  http_response_code(500);
  echo json_encode(['ok'=>false, 'error'=>'Failed to save']);
  exit;
}

// Normalisasi orientasi (khusus JPEG) dan stripping metadata dasar
try {
  if ($mime === 'image/jpeg' && function_exists('exif_read_data')) {
    $exif = @exif_read_data($dest);
    $orientation = $exif['Orientation'] ?? 1;
    if ($orientation != 1) {
      $image = imagecreatefromjpeg($dest);
      switch ($orientation) {
        case 2: imageflip($image, IMG_FLIP_HORIZONTAL); break;
        case 3: $image = imagerotate($image, 180, 0); break;
        case 4: imageflip($image, IMG_FLIP_VERTICAL); break;
        case 5: $image = imagerotate($image, -90, 0); imageflip($image, IMG_FLIP_HORIZONTAL); break;
        case 6: $image = imagerotate($image, -90, 0); break;
        case 7: $image = imagerotate($image, 90, 0); imageflip($image, IMG_FLIP_HORIZONTAL); break;
        case 8: $image = imagerotate($image, 90, 0); break;
      }
      // Tulis ulang JPEG untuk buang EXIF orientation
      imagejpeg($image, $dest, 92);
      imagedestroy($image);
    }
  } elseif ($mime === 'image/png') {
    // Optional: re-encode untuk hilangkan metadata besar
    $img = imagecreatefrompng($dest);
    imagesavealpha($img, true);
    imagepng($img, $dest, 6);
    imagedestroy($img);
  } elseif ($mime === 'image/webp') {
    $img = imagecreatefromwebp($dest);
    imagewebp($img, $dest, 80);
    imagedestroy($img);
  }
} catch (Throwable $e) {
  // Jika normalisasi gagal, biarkan file asli tetap tersimpan
}

$url = $publicPrefix . '/' . $fname;
echo json_encode(['ok'=>true, 'file'=>$fname, 'url'=>$url]);
